<?php
namespace VM\ApiBundle\Security\WebService;

use Symfony\Component\Security\Core\User\UserProviderInterface;
use Symfony\Component\Security\Core\User\UserInterface;
use Symfony\Component\Security\Core\Exception\UsernameNotFoundException;
use Symfony\Component\Security\Core\Exception\UnsupportedUserException;
use Symfony\Component\Security\Core\Exception\AuthenticationException;

use Symfony\Component\Validator\Constraints\Email;
use Symfony\Component\HttpFoundation\Request;

use VM\AdminBundle\Services\TokensGenerator;
use VM\ApiBundle\Entity\User;
use VM\ApiBundle\Services\UserManager;
use Symfony\Component\DependencyInjection\Container;
use VM\ApiBundle\Security\WebService\PasswordEncoder;
use VM\ApiBundle\Services\SSOAuthorization;


class WebServiceUserProvider implements UserProviderInterface
{
    protected $Doctrine;
    protected $NewAgeAuthorization;

    /**
     * @var SSOAuthorization
     */
    protected $SsoAuthorization;
    /**
     * @var Request
     */
    protected $Request;

    /**
     * @var UserManager
     */
    protected $UserManager;

    /**
     * @var Container
     */
    protected $Container;

    /**
     * @var PasswordEncoder
     */
    protected $PasswordEncoder;

    /**
     * @var TokensGenerator
     */
    protected $TokenGenerator;

    public function __construct(Container $Container)
    {
        $this->Doctrine = $Container->get('doctrine');
        $this->SsoAuthorization = $Container->get('sso.authorization');
        $this->UserManager = $Container->get('vm.user.manager');
        $this->PasswordEncoder = $Container->get('security_password_encoder');
        $this->Container = $Container;
        $this->TokenGenerator = $Container->get('vm_admin.tokens.generator');
    }


    public function loadUserByUsername($username)
    {
        $checkToken = $this->Container->get('security.context')->getToken();
        if(!empty($checkToken) && ($checkToken->getUser() instanceof User) && $checkToken->getUser()->getId())
        {
            return $checkToken->getUser();
        }

        $request = $this->getRequest();
        $password = $request->get('_password');

        $userDataFromSso = $this->authenticateUserInSso($username, $password);

        $User = $this->getUserByExternalId($userDataFromSso['user_id']);

        if(empty($User))
        {
//            $User = $this->UserManager->createUser($data['username'], $_POST['_password'], false, $fromNeDatabase);
            $User = $this->UserManager->createUser($username, $userDataFromSso);
        }

        $this->getRequest()->getSession()->set('user_sso_token', $userDataFromSso['token']);

        return $User;
    }

    public function refreshUser(UserInterface $user)
    {
        if (!$user instanceof User) {
            throw new UnsupportedUserException(sprintf('Instances of "%s" are not supported.', get_class($user)));
        }

        return $this->getUserByUsername($user->getUsername());
    }

    public function supportsClass($class)
    {
        return $class === 'VM\ApiBundle\Entity\User';
    }


    protected function getUserByUsername($username)
    {
        return $this->Doctrine->getRepository('VM\ApiBundle\Entity\User')->findOneByUsername($username);
    }

    protected function isValidUsername($email)
    {
        return preg_match('/[a-zA-Z\.\_0-9]+@[a-zA-Z\.\_0-9]+\.[a-zA-Z\.\_0-9]+/', $email);
    }

    private function changePasswordForUser(User $User, $newPass)
    {
        $this->UserManager->setUserPassword($User, $newPass);
        $this->Doctrine->getEntityManager()->persist($User);
        $this->Doctrine->getEntityManager()->flush();
        $this->Doctrine->getEntityManager()->refresh($User);
    }

    private function prepareUserData(User $User)
    {
        return array(
            'username' => $User->getUsername(),
            'password' => $User->getPassword(),
            'salt' => $User->getSalt(),
            'roles' => $User->getRoles()
        );
    }

    /**
     * @return Request
     */
    protected function getRequest()
    {
        return $this->Container->get('request');
    }

    protected function authenticateUserInSso($login, $password)
    {
        try
        {
            $data = $this->SsoAuthorization->authenticate($login, $password);
            return $data;
        }
        catch (\Exception $e)
        {
            throw new AuthenticationException($e->getMessage(), $e->getCode());
        }
    }

    protected function getUserByExternalId($external_user_id)
    {
        return $this->Doctrine->getRepository('VM\ApiBundle\Entity\User')->findOneBy(array('external_id' => $external_user_id));
    }
}